"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const electron_1 = require("electron");
const electron_2 = require("@capacitor-community/electron");
const native_lib_content_filtering_1 = require("@wontok/native-lib-content-filtering");
const native_lib_process_scanning_1 = require("@wontok/native-lib-process-scanning");
const os_1 = tslib_1.__importDefault(require("os"));
const electron_updater_1 = require("electron-updater");
const minimumNotificationInterval = 10; //seconds
const start = () => {
    console.log("App is starting");
    //Set theme to always be light
    electron_1.nativeTheme.themeSource = 'light';
    // The MainWindow object can be accessed via myCapacitorApp.getMainWindow()
    const myCapacitorApp = electron_2.createCapacitorElectronApp({
        applicationMenuTemplate: null,
        mainWindow: {
            windowOptions: {
                webPreferences: {
                    webSecurity: false
                },
                width: 768,
                height: 546
            }
        }
    });
    console.log(require('electron').app.getPath('userData'));
    // https://github.com/electron/electron/issues/23664#issuecomment-631674094
    electron_1.app.commandLine.appendSwitch('disable-features', 'OutOfBlinkCors');
    electron_1.app.setLoginItemSettings({
        openAtLogin: true
    });
    //Set custom protocol for deep linking
    electron_2.createCapacitorElectronDeepLinking(myCapacitorApp, {
        customProtocol: 'safecentralbyaol',
    });
    // Here's what I'm doing for a the quickest possible integration of the process scanning lib:
    // 2. When the state of the content filtering lib is changing, I'm propagating those changes to the process scanning lib, if initialized.
    // 3. I'm not sending any process scanning information to the renderer.
    // Important notes:
    // - Renderer is not aware of the fact that the process scanning lib exists.
    // - Hence, those changes that I'm propagating are supposed to just work - there will be no user feedback if they don't. 
    // To Note: On MacOS, the process scanning lib is not available until the content filtering extesion becomes available.
    // Hence, attempting to initialize and activate the process scanning lib would result in a XPC connection initialization error.
    // Hence, we must initialize the process scanning lib only after the the content filtering lib has been initialized and made available.
    // I need to keep a reference to them
    let processScanningLib = undefined;
    let contentFilteringLib = undefined;
    let tray;
    let trayContextMenu = electron_1.Menu.buildFromTemplate([
        {
            label: 'Show App', click: function () {
                myCapacitorApp.getMainWindow().show();
                myCapacitorApp.getMainWindow().reload();
            }
        },
        {
            id: "launch-browser",
            label: 'Launch Browser', click: function () {
                contentFilteringLib.launchSafeBrowsing();
            }
        },
        // The visibility of the following items depends on the license status and protection status.
        {
            id: "invalid-license",
            visible: false,
            enabled: false,
            label: "Invalid License! Show App to update status.",
            click: function () {
                // Always disabled.
            }
        },
    ]);
    const unpropagated = [];
    let propagatingRightNow = false;
    function propagateTo(processScanningLib) {
        return () => {
            if (!propagatingRightNow) {
                propagatingRightNow = true;
                // Remove and process each state in order.
                while (unpropagated.length > 0) {
                    const newState = unpropagated.shift();
                    // propagate the "working/not-working" portion.
                    if (newState.isFiltering) {
                        processScanningLib.startScanning();
                        contentFilteringLib.launchGhostpress();
                    }
                    else {
                        processScanningLib.stopScanning();
                        contentFilteringLib.terminateGhostpress();
                    }
                    // propagate the license expiration timestamp
                    processScanningLib.setLicenseExpirationDate(newState.licenceExpirationDate);
                    const currentDate = new Date();
                    trayContextMenu.getMenuItemById("invalid-license").visible = newState.licenceExpirationDate < currentDate;
                    trayContextMenu.getMenuItemById("launch-browser").visible = newState.licenceExpirationDate >= currentDate;
                }
                propagatingRightNow = false;
            }
        };
    }
    function startPropagatingTo(processScanningLib) {
        setInterval(propagateTo(processScanningLib), 30000);
    }
    function schedule(newState) {
        unpropagated.push(newState);
    }
    const setupProcessScanningLib = (newState, firstTime) => {
        if (firstTime) {
            processScanningLib = native_lib_process_scanning_1.initLib({
                onStateChange: (newState) => {
                    // trayContextMenu.getMenuItemById("enable-background-protection").visible = !newState.isScanning;
                    // trayContextMenu.getMenuItemById("disable-background-protection").visible = newState.isScanning;
                },
                onBlockedContent: (blockedContentInfo) => {
                    new electron_1.Notification({
                        title: `Killed process`,
                        body: `${blockedContentInfo.description} - ${blockedContentInfo.reason}`,
                    }).show();
                }
            });
            processScanningLib.activateNativeService();
            startPropagatingTo(processScanningLib);
            setupProcessScanningLib(newState, false);
        }
        else {
            // 2. When the state of the content filtering lib is changing, I'm propagating those changes to the process scanning lib
            schedule(newState);
        }
    };
    let sentNotifications = [];
    let checkForUpdateStarted = false;
    // The Content Filtering Lib object
    contentFilteringLib = native_lib_content_filtering_1.initLib({
        onStateChange: (newState) => {
            if (newState.nativeServiceAvailable) {
                if (newState.blockedCategories.length === 0) {
                    contentFilteringLib.setBlockedCategories([
                        "malware",
                        "phishing",
                        "fraud",
                        "pua",
                        "miner",
                    ]);
                }
                if (!checkForUpdateStarted) {
                    checkForUpdateStarted = true;
                    // Whenever an update has been downloaded, just install it.
                    electron_updater_1.autoUpdater.on("update-downloaded", (updateDownloaded) => {
                        if (process.platform === "darwin") {
                            const updateFilePath = updateDownloaded.downloadedFile;
                            console.log("Dispatching MacOS update downloaded event to the native extension:", updateFilePath);
                            contentFilteringLib.updateTo(updateFilePath);
                            console.log("Quit - the extension will restart us once the update is done.");
                            electron_1.app.quit();
                        }
                        else {
                            electron_updater_1.autoUpdater.quitAndInstall(false, true);
                        }
                    });
                    // Check for updates on startup.
                    electron_updater_1.autoUpdater.checkForUpdates();
                    // And then check every 15 minutes
                    setInterval(() => {
                        electron_updater_1.autoUpdater.checkForUpdates();
                    }, 1000 * 60 * 15);
                    // This is basically the first state, so now it is safe to setup the process scanning lib.
                    setupProcessScanningLib(newState, true);
                }
                else {
                    setupProcessScanningLib(newState, false);
                }
            }
        },
        onBlockedContent: (blockedContentInfo) => {
            const currentTimestamp = Math.floor(Date.now() / 1000);
            sentNotifications = sentNotifications.filter((notification) => currentTimestamp - notification.timestamp < minimumNotificationInterval);
            const alreadySent = sentNotifications.find((notification) => notification.url === blockedContentInfo.url);
            if (alreadySent === undefined) {
                new electron_1.Notification({
                    title: `Blocked ${blockedContentInfo.url}`,
                    body: blockedContentInfo.categories.join(),
                }).show();
                sentNotifications.push({
                    url: blockedContentInfo.url,
                    timestamp: currentTimestamp
                });
            }
        }
    });
    // This will trigger the first state change!
    // We're going to run it some 10 seconds later just to allow the app to show the loading screen.
    setTimeout(contentFilteringLib.activateNativeService, 5000);
    let isQuiting;
    // This method will be called when Electron has finished
    // initialization and is ready to create browser windows.
    // Some Electron APIs can only be used after this event occurs.
    electron_1.app.on("ready", () => {
        myCapacitorApp.init();
        if (os_1.default.type() === "Windows_NT") {
            electron_1.app.setAppUserModelId("com.wontok.safecentralaol");
        }
        // Make urls open in extenal browsers
        myCapacitorApp.getMainWindow().webContents.on('new-window', function (e, url) {
            e.preventDefault();
            electron_1.shell.openExternal(url);
        });
        myCapacitorApp.getMainWindow().on('close', function (event) {
            if (!isQuiting) {
                event.preventDefault();
                myCapacitorApp.getMainWindow().hide();
                event.returnValue = false;
            }
        });
        if (os_1.default.type() === 'Windows_NT') {
            tray = new electron_1.Tray(electron_1.app.getAppPath() + '/assets/appIcon.png');
        }
        else {
            tray = new electron_1.Tray(electron_1.nativeImage.createFromPath(electron_1.app.getAppPath() + '/assets/appIcon.png').resize({ width: 16, height: 16 }));
        }
        tray.setContextMenu(trayContextMenu);
        tray.on('click', function () {
            myCapacitorApp.getMainWindow().show();
            myCapacitorApp.getMainWindow().reload();
        });
    });
    // Quit when all windows are closed.
    electron_1.app.on("window-all-closed", function () {
        // On OS X it is common for applications and their menu bar
        // to stay active until the user quits explicitly with Cmd + Q
        if (process.platform !== "darwin") {
            electron_1.app.quit();
        }
    });
    electron_1.app.on("activate", function () {
        // On OS X it's common to re-create a window in the app when the
        // dock icon is clicked and there are no other windows open.
        if (myCapacitorApp.getMainWindow().isDestroyed())
            myCapacitorApp.init();
    });
    electron_1.app.on('before-quit', function () {
        isQuiting = true;
    });
    electron_1.app.on('second-instance', () => {
        myCapacitorApp.getMainWindow().show();
        myCapacitorApp.getMainWindow().reload();
    });
    // Define any IPC or other custom functionality below here
};
console.log("Os Type:", os_1.default.type());
console.log("App is packaged", electron_1.app.isPackaged);
console.log("App path", electron_1.app.getAppPath());
start();
